<?php
declare(strict_types=1);
require __DIR__ . '/lib/telegram.php';
require __DIR__ . '/lib/steam.php';
require __DIR__ . '/lib/limits_core.php';

$cfg = env_cfg();
$DOMAIN = rtrim((string)$cfg['DOMAIN'], '/');
$token = $_GET['bot'] ?? '';
if (!$token) { http_response_code(200); exit('OK'); }

try {
    $db = pdo();
} catch (PDOException $e) {
    error_log("PDO Error in sub_bot.php: " . $e->getMessage());
    http_response_code(500);
    exit;
}
// Load bot settings
$stmt = $db->prepare("SELECT * FROM sub_bots WHERE bot_token=? LIMIT 1");
$stmt->execute([$token]);
$bot = $stmt->fetch();
if (!$bot) { http_response_code(200); exit('OK'); }

// Store bot ID for groups system
$current_bot_id = $bot['id'];

$update = json_decode(file_get_contents('php://input') ?: '[]', true);
if (!$update) { http_response_code(200); exit('OK'); }

$msg = $update['message'] ?? $update['edited_message'] ?? null;
if (!$msg) { http_response_code(200); exit('OK'); }
$user_id = $msg['from']['id'] ?? 0;
$chat_id = $msg['chat']['id'] ?? 0;
$text = trim($msg['text'] ?? '');

if ($bot['is_paused']) {
    $paused = template_get('paused');
    tg_send($token, $chat_id, $paused);
    exit;
}

// Ensure allowed user / max users / subscription
$stmt = $db->prepare("SELECT * FROM allowed_users WHERE bot_token=? AND user_id=? LIMIT 1");
$stmt->execute([$token, $user_id]);
$allowed = $stmt->fetch();

if (!$allowed) {
    // Check max users
    $cnt = $db->prepare("SELECT COUNT(*) c FROM allowed_users WHERE bot_token=? AND banned=0");
    $cnt->execute([$token]);
    $current = (int)$cnt->fetch()['c'];
    if ($current >= (int)$bot['max_users']) {
        tg_send($token, $chat_id, template_get('max_users_reached'));
        exit;
    }
    // Check subscription
    $channel_id = $bot['channel_id'];
    if ($channel_id && !is_subscribed($token, $channel_id, $user_id)) {
        tg_send($token, $chat_id, template_get('not_subscribed'));
        exit;
    }
    // Allow user
    $db->prepare("INSERT IGNORE INTO allowed_users(bot_token,user_id) VALUES(?,?)")->execute([$token, $user_id]);
    $allowed = ['banned'=>0];
} else {
    if ((int)$allowed['banned'] === 1) {
        tg_send($token, $chat_id, "🚫 تم حظرك من استخدام هذا البوت.");
        exit;
    }
}

if ($text === '/start') {
    $welcome = $bot['welcome_message'] ?: (template_get('start_1') . "\n" . template_get('start_2'));
    tg_send($token, $chat_id, $welcome);
    exit;
}

// ============================================
// أوامر المدير الفرعي (Sub-Manager Commands)
// ============================================
if (strpos($text, '/') === 0) {
    // التحقق من أن المستخدم مدير لهذا البوت
    $manager = get_manager($user_id, $token);
    
    if (!$manager) {
        // ليس مديراً، تجاهل الأمر وعامله كاسم حساب
        goto treat_as_account_name;
    }
    
    // معالجة الأوامر
    $parts = preg_split('/\s+/', $text, 2);
    $cmd = $parts[0];
    $arg = $parts[1] ?? '';
    
    switch ($cmd) {
        // ============================================
        // 📋 قائمة أوامر المدير
        // ============================================
        
        case '/admin': {
            // بناء قائمة الأوامر حسب الصلاحيات
            $perms = $manager['permissions'] ?? '';
            $is_all = strtolower(trim($perms)) === 'all';
            
            $out = "🎛 <b>لوحة تحكم المدير الفرعي</b>\n\n";
            $out .= "👤 معرفك: <code>{$user_id}</code>\n";
            $out .= "🔐 صلاحياتك: <b>" . ($is_all ? 'كل الصلاحيات ✨' : $perms) . "</b>\n\n";
            $out .= "━━━━━━━━━━━━━━━━━━\n\n";
            
            // تحديد الصلاحيات
            $has = function($perm_id) use ($user_id, $token) {
                return has_permission($user_id, $perm_id, $token);
            };
            
            // 👥 إدارة المستخدمين
            if ($has(4) || $has(5) || $has(2)) {
                $out .= "👥 <b>إدارة المستخدمين:</b>\n\n";
                
                if ($has(4)) {
                    $out .= "📋 <b>/users</b>\n";
                    $out .= "   └ عرض جميع مستخدمي البوت (آخر 100)\n";
                    $out .= "   └ مثال: <code>/users</code>\n\n";
                }
                
                if ($has(5)) {
                    $out .= "🔍 <b>/search</b> <code>&lt;user_id&gt;</code>\n";
                    $out .= "   └ البحث عن مستخدم معين\n";
                    $out .= "   └ مثال: <code>/search 123456789</code>\n\n";
                }
                
                if ($has(2)) {
                    $out .= "🚫 <b>/ban</b> <code>&lt;user_id&gt;</code>\n";
                    $out .= "   └ حظر مستخدم من البوت\n";
                    $out .= "   └ مثال: <code>/ban 123456789</code>\n\n";
                    
                    $out .= "✅ <b>/unban</b> <code>&lt;user_id&gt;</code>\n";
                    $out .= "   └ إلغاء حظر مستخدم\n";
                    $out .= "   └ مثال: <code>/unban 123456789</code>\n\n";
                }
                
                if ($has(4)) {
                    $out .= "🏆 <b>/top_users</b> <code>[limit]</code>\n";
                    $out .= "   └ أكثر المستخدمين نشاطاً (آخر 7 أيام)\n";
                    $out .= "   └ مثال: <code>/top_users 10</code>\n\n";
                }
                
                $out .= "━━━━━━━━━━━━━━━━━━\n\n";
            }
            
            // 📢 البث
            if ($has(1)) {
                $out .= "📢 <b>البث الجماعي:</b>\n\n";
                
                $out .= "📣 <b>/broadcast</b> <code>&lt;نص&gt;</code>\n";
                $out .= "   └ إرسال رسالة لجميع المستخدمين\n";
                $out .= "   └ مثال: <code>/broadcast مرحباً بالجميع!</code>\n\n";
                
                $out .= "📜 <b>/broadcast_log</b>\n";
                $out .= "   └ عرض سجل آخر 10 عمليات بث\n";
                $out .= "   └ مثال: <code>/broadcast_log</code>\n\n";
                
                $out .= "━━━━━━━━━━━━━━━━━━\n\n";
            }
            
            // 💳 القسائم
            if ($has(3)) {
                $out .= "💳 <b>إدارة القسائم:</b>\n\n";
                
                $out .= "🎫 <b>/redeem</b> <code>&lt;code&gt;</code>\n";
                $out .= "   └ تعبئة قسيمة في البوت\n";
                $out .= "   └ مثال: <code>/redeem CODE-12345</code>\n\n";
                
                $out .= "📜 <b>/vouchers_log</b>\n";
                $out .= "   └ عرض سجل آخر 20 قسيمة\n";
                $out .= "   └ مثال: <code>/vouchers_log</code>\n\n";
                
                $out .= "━━━━━━━━━━━━━━━━━━\n\n";
            }
            
            // 📊 الإحصائيات
            if ($has(4) || $has(5)) {
                $out .= "📊 <b>الإحصائيات والتقارير:</b>\n\n";
                
                if ($has(4)) {
                    $out .= "📈 <b>/mystats</b>\n";
                    $out .= "   └ ملخص نشاط البوت\n";
                    $out .= "   └ مثال: <code>/mystats</code>\n\n";
                    
                    $out .= "🎮 <b>/codes_today</b>\n";
                    $out .= "   └ عدد الأكواد المسحوبة اليوم\n";
                    $out .= "   └ مثال: <code>/codes_today</code>\n\n";
                    
                    $out .= "🧮 <b>/quota</b>\n";
                    $out .= "   └ رصيد البوت (المستخدمون/الحد الأقصى)\n";
                    $out .= "   └ مثال: <code>/quota</code>\n\n";
                }
                
                if ($has(5)) {
                    $out .= "👤 <b>/userstats</b> <code>&lt;user_id&gt;</code>\n";
                    $out .= "   └ إحصائيات مستخدم معين\n";
                    $out .= "   └ مثال: <code>/userstats 123456789</code>\n\n";
                }
                
                $out .= "━━━━━━━━━━━━━━━━━━\n\n";
            }
            
            // تذييل
            $out .= "💡 <b>ملاحظات:</b>\n";
            $out .= "• جميع الأوامر تعمل في هذا البوت فقط\n";
            $out .= "• استخدم <code>/admin</code> لعرض هذه القائمة\n";
            $out .= "• الصلاحيات يحددها المالك من البوت المركزي\n\n";
            
            $out .= "🔐 <b>شرح الصلاحيات:</b>\n";
            $out .= "1️⃣ = بث | 2️⃣ = حظر | 3️⃣ = قسائم\n";
            $out .= "4️⃣ = مستخدمين | 5️⃣ = بحث | 6️⃣ = حدود";
            
            tg_send($token, $chat_id, $out);
            exit;
        }
        
        // ============================================
        // 👥 إدارة المستخدمين (5 أوامر)
        // ============================================
        
        case '/users': {
            if (!has_permission($user_id, 4, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية عرض المستخدمين");
                exit;
            }
            
            $users = $db->prepare("SELECT user_id, banned, created_at FROM allowed_users WHERE bot_token=? ORDER BY id DESC LIMIT 100");
            $users->execute([$token]);
            $list = $users->fetchAll();
            
            if (!$list) {
                tg_send($token, $chat_id, "لا يوجد مستخدمون");
                exit;
            }
            
            $out = "👥 <b>مستخدمو البوت:</b>\n\n";
            $active = 0; $banned = 0;
            foreach ($list as $u) {
                $status = (int)$u['banned'] === 1 ? '🚫' : '✅';
                $out .= "{$status} <code>{$u['user_id']}</code> - {$u['created_at']}\n";
                if ((int)$u['banned'] === 1) $banned++; else $active++;
            }
            $out .= "\n✅ نشط: {$active} | 🚫 محظور: {$banned}";
            
            tg_send($token, $chat_id, $out);
            exit;
        }
        
        case '/search': {
            if (!has_permission($user_id, 5, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية البحث");
                exit;
            }
            
            if (!$arg) {
                tg_send($token, $chat_id, "الاستخدام: /search <user_id>");
                exit;
            }
            
            $search_id = (int)$arg;
            $u = $db->prepare("SELECT * FROM allowed_users WHERE bot_token=? AND user_id=? LIMIT 1");
            $u->execute([$token, $search_id]);
            $user = $u->fetch();
            
            if (!$user) {
                tg_send($token, $chat_id, "❌ المستخدم غير موجود في هذا البوت");
                exit;
            }
            
            // عدد الأكواد المسحوبة
            $codes = $db->prepare("SELECT COUNT(*) c FROM steam_requests WHERE bot_token=? AND user_id=?");
            $codes->execute([$token, $search_id]);
            $total_codes = (int)$codes->fetch()['c'];
            
            $status = (int)$user['banned'] === 1 ? '🚫 محظور' : '✅ نشط';
            
            $out = "🔍 <b>نتيجة البحث:</b>\n\n";
            $out .= "👤 ID: <code>{$search_id}</code>\n";
            $out .= "📊 الحالة: {$status}\n";
            $out .= "🎮 الأكواد المسحوبة: {$total_codes}\n";
            $out .= "📅 تاريخ الانضمام: {$user['created_at']}";
            
            tg_send($token, $chat_id, $out);
            exit;
        }
        
        case '/ban': {
            if (!has_permission($user_id, 2, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية الحظر");
                exit;
            }
            
            if (!$arg) {
                tg_send($token, $chat_id, "الاستخدام: /ban <user_id>");
                exit;
            }
            
            $ban_id = (int)$arg;
            $db->prepare("UPDATE allowed_users SET banned=1 WHERE bot_token=? AND user_id=?")->execute([$token, $ban_id]);
            
            tg_send($token, $chat_id, "✅ <b>تم حظر المستخدم</b>\n\n👤 ID: <code>{$ban_id}</code>");
            exit;
        }
        
        case '/unban': {
            if (!has_permission($user_id, 2, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية إلغاء الحظر");
                exit;
            }
            
            if (!$arg) {
                tg_send($token, $chat_id, "الاستخدام: /unban <user_id>");
                exit;
            }
            
            $unban_id = (int)$arg;
            $db->prepare("UPDATE allowed_users SET banned=0 WHERE bot_token=? AND user_id=?")->execute([$token, $unban_id]);
            
            tg_send($token, $chat_id, "✅ <b>تم إلغاء حظر المستخدم</b>\n\n👤 ID: <code>{$unban_id}</code>");
            exit;
        }
        
        case '/top_users': {
            if (!has_permission($user_id, 4, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية عرض المستخدمين");
                exit;
            }
            
            $limit = (int)($arg ?: 10);
            $since = date('Y-m-d H:i:s', time() - (7 * 86400)); // آخر 7 أيام
            
            $top = $db->prepare("
                SELECT user_id, COUNT(*) as cnt 
                FROM steam_requests 
                WHERE bot_token=? AND request_time >= ?
                GROUP BY user_id 
                ORDER BY cnt DESC 
                LIMIT ?
            ");
            $top->execute([$token, $since, $limit]);
            $list = $top->fetchAll();
            
            if (!$list) {
                tg_send($token, $chat_id, "لا توجد بيانات");
                exit;
            }
            
            $out = "🏆 <b>أكثر المستخدمين نشاطاً (آخر 7 أيام):</b>\n\n";
            $rank = 1;
            foreach ($list as $u) {
                $medal = $rank === 1 ? '🥇' : ($rank === 2 ? '🥈' : ($rank === 3 ? '🥉' : "{$rank}️⃣"));
                $out .= "{$medal} <code>{$u['user_id']}</code> - {$u['cnt']} كود\n";
                $rank++;
            }
            
            tg_send($token, $chat_id, $out);
            exit;
        }
        
        // ============================================
        // 📢 البث (3 أوامر)
        // ============================================
        
        case '/broadcast': {
            if (!has_permission($user_id, 1, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية البث");
                exit;
            }
            
            if (!$arg) {
                tg_send($token, $chat_id, "الاستخدام: /broadcast <نص الرسالة>");
                exit;
            }
            
            $users = $db->prepare("SELECT DISTINCT user_id FROM allowed_users WHERE bot_token=? AND banned=0");
            $users->execute([$token]);
            $list = $users->fetchAll(PDO::FETCH_COLUMN);
            
            $success = 0; $failed = 0;
            foreach ($list as $uid) {
                $result = tg_send($token, $uid, $arg);
                if ($result) $success++; else $failed++;
                usleep(50000); // 50ms تأخير
            }
            
            // تسجيل البث
            $db->prepare("INSERT INTO broadcast_log(bot_token, manager_id, message_text, recipients_count) VALUES(?,?,?,?)")
                ->execute([$token, $user_id, substr($arg, 0, 500), $success]);
            
            tg_send($token, $chat_id, "✅ <b>اكتمل البث</b>\n\n✅ نجح: {$success}\n❌ فشل: {$failed}");
            exit;
        }
        
        case '/broadcast_media': {
            if (!has_permission($user_id, 1, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية البث");
                exit;
            }
            
            tg_send($token, $chat_id, "⚠️ هذا الأمر قيد التطوير");
            exit;
        }
        
        case '/broadcast_log': {
            if (!has_permission($user_id, 1, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية عرض سجل البث");
                exit;
            }
            
            $logs = $db->prepare("SELECT * FROM broadcast_log WHERE bot_token=? AND manager_id=? ORDER BY id DESC LIMIT 10");
            $logs->execute([$token, $user_id]);
            $list = $logs->fetchAll();
            
            if (!$list) {
                tg_send($token, $chat_id, "لا يوجد سجل بث");
                exit;
            }
            
            $out = "📜 <b>سجل البث:</b>\n\n";
            foreach ($list as $l) {
                $preview = substr($l['message_text'], 0, 50) . '...';
                $out .= "📅 {$l['created_at']}\n";
                $out .= "👥 {$l['recipients_count']} مستلم\n";
                $out .= "📝 {$preview}\n\n";
            }
            
            tg_send($token, $chat_id, $out);
            exit;
        }
        
        // ============================================
        // 💳 القسائم (2 أمر)
        // ============================================
        
        case '/redeem': {
            if (!has_permission($user_id, 3, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية تعبئة القسائم");
                exit;
            }
            
            if (!$arg) {
                tg_send($token, $chat_id, "الاستخدام: /redeem <code>");
                exit;
            }
            
            $code = strtoupper($arg);
            $v = $db->prepare("SELECT * FROM vouchers WHERE code=? LIMIT 1");
            $v->execute([$code]);
            $voucher = $v->fetch();
            
            if (!$voucher) {
                tg_send($token, $chat_id, "❌ القسيمة غير موجودة");
                exit;
            }
            
            if ($voucher['current_uses'] >= $voucher['max_uses']) {
                tg_send($token, $chat_id, "❌ تم استهلاك القسيمة");
                exit;
            }
            
            if ($voucher['voucher_type'] === 'users') {
                $db->prepare("UPDATE sub_bots SET max_users = max_users + ? WHERE bot_token=?")->execute([$voucher['amount'], $token]);
            }
            
            $db->prepare("UPDATE vouchers SET current_uses = current_uses + 1 WHERE code=?")->execute([$code]);
            
            // تسجيل استخدام القسيمة
            $db->prepare("INSERT INTO voucher_redemptions(voucher_code, bot_token, redeemed_by, amount_added) VALUES(?,?,?,?)")
                ->execute([$code, $token, $user_id, $voucher['amount']]);
            
            tg_send($token, $chat_id, "✅ <b>تم تفعيل القسيمة</b>\n\n💰 تمت إضافة: {$voucher['amount']} مستخدم");
            exit;
        }
        
        case '/vouchers_log': {
            if (!has_permission($user_id, 3, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية عرض سجل القسائم");
                exit;
            }
            
            $logs = $db->prepare("SELECT * FROM voucher_redemptions WHERE bot_token=? ORDER BY id DESC LIMIT 20");
            $logs->execute([$token]);
            $list = $logs->fetchAll();
            
            if (!$list) {
                tg_send($token, $chat_id, "لا يوجد سجل قسائم");
                exit;
            }
            
            $out = "📜 <b>سجل القسائم:</b>\n\n";
            $num = 1;
            foreach ($list as $l) {
                $out .= "{$num}️⃣ <code>{$l['voucher_code']}</code> | ✅ مستخدمة\n";
                $out .= "   📅 {$l['redeemed_at']} | +{$l['amount_added']} مستخدم\n\n";
                $num++;
            }
            
            tg_send($token, $chat_id, $out);
            exit;
        }
        
        // ============================================
        // 📊 الإحصائيات (4 أوامر)
        // ============================================
        
        case '/mystats': {
            if (!has_permission($user_id, 4, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية عرض الإحصائيات");
                exit;
            }
            
            // عدد المستخدمين
            $users_cnt = $db->prepare("SELECT COUNT(*) c FROM allowed_users WHERE bot_token=? AND banned=0");
            $users_cnt->execute([$token]);
            $users = (int)$users_cnt->fetch()['c'];
            
            // الأكواد اليوم
            $today = date('Y-m-d 00:00:00');
            $codes_today = $db->prepare("SELECT COUNT(*) c FROM steam_requests WHERE bot_token=? AND request_time >= ?");
            $codes_today->execute([$token, $today]);
            $today_cnt = (int)$codes_today->fetch()['c'];
            
            // الأكواد هذا الأسبوع
            $week_ago = date('Y-m-d H:i:s', time() - (7 * 86400));
            $codes_week = $db->prepare("SELECT COUNT(*) c FROM steam_requests WHERE bot_token=? AND request_time >= ?");
            $codes_week->execute([$token, $week_ago]);
            $week_cnt = (int)$codes_week->fetch()['c'];
            
            $out = "📈 <b>إحصاءات بوتك:</b>\n\n";
            $out .= "👥 عدد المستخدمين الحاليين: <b>{$users}</b>\n";
            $out .= "🎯 الحد الأقصى للمستخدمين: <b>{$bot['max_users']}</b>\n";
            $out .= "🎮 الأكواد المسحوبة اليوم: <b>{$today_cnt}</b>\n";
            $out .= "📊 الأكواد المسحوبة هذا الأسبوع: <b>{$week_cnt}</b>";
            
            tg_send($token, $chat_id, $out);
            exit;
        }
        
        case '/userstats': {
            if (!has_permission($user_id, 5, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية البحث");
                exit;
            }
            
            if (!$arg) {
                tg_send($token, $chat_id, "الاستخدام: /userstats <user_id>");
                exit;
            }
            
            $target_id = (int)$arg;
            
            // اليوم
            $today = date('Y-m-d 00:00:00');
            $t = $db->prepare("SELECT COUNT(*) c FROM steam_requests WHERE bot_token=? AND user_id=? AND request_time >= ?");
            $t->execute([$token, $target_id, $today]);
            $today_cnt = (int)$t->fetch()['c'];
            
            // 7 أيام
            $week = date('Y-m-d H:i:s', time() - (7 * 86400));
            $w = $db->prepare("SELECT COUNT(*) c FROM steam_requests WHERE bot_token=? AND user_id=? AND request_time >= ?");
            $w->execute([$token, $target_id, $week]);
            $week_cnt = (int)$w->fetch()['c'];
            
            // 30 يوم
            $month = date('Y-m-d H:i:s', time() - (30 * 86400));
            $m = $db->prepare("SELECT COUNT(*) c FROM steam_requests WHERE bot_token=? AND user_id=? AND request_time >= ?");
            $m->execute([$token, $target_id, $month]);
            $month_cnt = (int)$m->fetch()['c'];
            
            $out = "📊 <b>إحصائيات المستخدم</b>\n\n";
            $out .= "👤 ID: <code>{$target_id}</code>\n\n";
            $out .= "📅 اليوم: <b>{$today_cnt}</b> كود\n";
            $out .= "📅 آخر 7 أيام: <b>{$week_cnt}</b> كود\n";
            $out .= "📅 آخر 30 يوم: <b>{$month_cnt}</b> كود";
            
            tg_send($token, $chat_id, $out);
            exit;
        }
        
        case '/codes_today': {
            if (!has_permission($user_id, 4, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية عرض الإحصائيات");
                exit;
            }
            
            $today = date('Y-m-d 00:00:00');
            $codes = $db->prepare("SELECT COUNT(*) c FROM steam_requests WHERE bot_token=? AND request_time >= ?");
            $codes->execute([$token, $today]);
            $cnt = (int)$codes->fetch()['c'];
            
            tg_send($token, $chat_id, "🎮 <b>الأكواد المسحوبة اليوم:</b> {$cnt}");
            exit;
        }
        
        case '/quota': {
            if (!has_permission($user_id, 4, $token)) {
                tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية عرض الرصيد");
                exit;
            }
            
            $users_cnt = $db->prepare("SELECT COUNT(*) c FROM allowed_users WHERE bot_token=? AND banned=0");
            $users_cnt->execute([$token]);
            $current = (int)$users_cnt->fetch()['c'];
            $max = (int)$bot['max_users'];
            
            $percentage = $max > 0 ? round(($current / $max) * 100) : 0;
            $bar = str_repeat('▓', (int)($percentage / 10)) . str_repeat('░', 10 - (int)($percentage / 10));
            
            $out = "🧮 <b>رصيد البوت:</b>\n\n";
            $out .= "👥 المستخدمون الحاليون: <b>{$current}</b>\n";
            $out .= "🎯 الحد الأقصى: <b>{$max}</b>\n";
            $out .= "📊 النسبة: <b>{$percentage}%</b>\n\n";
            $out .= "{$bar}";
            
            tg_send($token, $chat_id, $out);
            exit;
        }
        
        default:
            // أمر غير معروف للمدير، عامله كاسم حساب
            goto treat_as_account_name;
    }
}

treat_as_account_name:
// Treat any other text as account name
// نظام المجموعات الجديد - البوت يرى فقط الحسابات في مجموعاته
$acc = $db->prepare("
    SELECT a.* 
    FROM steam_accounts a
    JOIN group_accounts ga ON ga.account_id = a.id
    JOIN bot_groups bg ON bg.group_id = ga.group_id
    JOIN account_groups g ON g.id = ga.group_id
    WHERE bg.bot_id = ? 
      AND a.account_name = ?
      AND a.active = 1
      AND g.active = 1
    LIMIT 1
");
$acc->execute([$current_bot_id, $text]);
$account = $acc->fetch();

if (!$account) {
    // رسالة موحدة للحماية - لا نكشف السبب الحقيقي
    tg_send($token, $chat_id, "❌ اسم الحساب غير صحيح أو غير مفعل.");
    exit;
}

// ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
// نظام الحدود الذكي (Smart Limit System)
// ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
$limit_check = check_and_consume_limit(
    $db,
    $bot['label'],
    $user_id,
    (int)$account['id'],
    $account['account_name']
);

if (!$limit_check['allowed']) {
    // مرفوض - إرسال رسالة الحد
    tg_send($token, $chat_id, $limit_check['message']);
    exit;
}

// مسموح - توليد الكود
$code = steam_guard_code($account['shared_secret']);

// تسجيل الطلب
$db->prepare("INSERT INTO steam_requests(bot_token,user_id,account_name) VALUES(?,?,?)")
   ->execute([$token, $user_id, $account['account_name']]);

// تحضير الرسالة
$replyTpl = template_get('code_reply');
$reply = interpolate($replyTpl, ['account'=>$account['account_name'], 'code'=>$code]);

// إضافة تنبيه إذا وُجد
if (!empty($limit_check['warning'])) {
    $reply .= "\n\n" . $limit_check['warning'];
}

tg_send($token, $chat_id, $reply);
http_response_code(200);
echo 'OK';
