# تقرير نظام القسائم (Vouchers System)

## 📋 نظرة عامة

نظام القسائم يتيح للمالك **إنشاء أكواد (Vouchers)** يمكن تفعيلها لزيادة قدرات البوتات أو منح امتيازات للمستخدمين.

**الإصدار الحالي**: v2.0

**الجداول المستخدمة**:
1. `vouchers` - القسائم المتاحة
2. `voucher_redemptions` - سجل التفعيل
3. `voucher_usage` - (قديم/مهمل)

---

## 🎫 جدول `vouchers` (القسائم)

### بنية الجدول:

```sql
CREATE TABLE `vouchers` (
  `id` bigint(20) PRIMARY KEY AUTO_INCREMENT,
  `code` varchar(32) UNIQUE NOT NULL,
  `voucher_type` enum('users','days','unlimited') DEFAULT 'users',
  `amount` int(11) NOT NULL,
  `target_label` varchar(64) DEFAULT NULL,
  `max_uses` int(11) DEFAULT 1,
  `current_uses` int(11) DEFAULT 0,
  `created_by_user_id` bigint(20) NOT NULL,
  `created_at` timestamp DEFAULT current_timestamp()
);
```

### شرح الحقول:

| الحقل | الوصف | مثال |
|------|-------|------|
| `code` | كود القسيمة (فريد) | `VA3F2B9C1D4E` |
| `voucher_type` | نوع القسيمة | `users`, `days`, `unlimited` |
| `amount` | القيمة/الكمية | 100 (مستخدمين) أو 30 (أيام) |
| `target_label` | البوت المستهدف (NULL = للجميع) | `test1` أو `NULL` |
| `max_uses` | عدد مرات الاستخدام المسموح | 1, 5, 10, ... |
| `current_uses` | عدد مرات الاستخدام الفعلية | 0, 1, 2, ... |
| `created_by_user_id` | المالك الذي أنشأها | `475567305` |

---

## 🎫 جدول `voucher_redemptions` (سجل التفعيل)

### بنية الجدول:

```sql
CREATE TABLE `voucher_redemptions` (
  `id` bigint(20) PRIMARY KEY AUTO_INCREMENT,
  `voucher_code` varchar(32) NOT NULL,
  `bot_token` varchar(128) NOT NULL,
  `redeemed_by` bigint(20) NOT NULL,
  `amount_added` int(11) NOT NULL,
  `redeemed_at` timestamp DEFAULT current_timestamp(),
  FOREIGN KEY (`bot_token`) REFERENCES `sub_bots`(`bot_token`) ON DELETE CASCADE
);
```

### شرح الحقول:

| الحقل | الوصف | مثال |
|------|-------|------|
| `voucher_code` | كود القسيمة المستخدمة | `VA3F2B9C1D4E` |
| `bot_token` | البوت الذي تم تفعيل القسيمة فيه | `123:ABC...` |
| `redeemed_by` | من فعّل القسيمة (مالك أو مدير) | `475567305` |
| `amount_added` | الكمية المضافة فعلياً | 100 |
| `redeemed_at` | تاريخ ووقت التفعيل | `2025-11-15 10:14:00` |

---

## 🏗️ أنواع القسائم

### 1. قسائم المستخدمين (`users`)

**الوظيفة**: زيادة `max_users` في البوت

**مثال**: قسيمة تضيف 100 مستخدم لبوت معين

**الاستخدام الحالي**: ✅ **مفعّل ويعمل**

---

### 2. قسائم الأيام (`days`)

**الوظيفة**: (غير مفعّل حالياً)

**الاستخدام المخطط**: زيادة صلاحية اشتراك أو تمديد خدمة

**الحالة**: ⚠️ **موجود في الكود لكن غير مطبق**

---

### 3. قسائم غير محدودة (`unlimited`)

**الوظيفة**: (غير مفعّل حالياً)

**الاستخدام المخطط**: إزالة جميع الحدود

**الحالة**: ⚠️ **موجود في الكود لكن غير مطبق**

---

## 🔨 إنشاء القسائم

### الأمر: `/generate_voucher`

**الملف**: `control_bot.php` السطر 877

**من يستطيع**: المالك فقط

### الصيغة:

```bash
/generate_voucher <type> <amount> [target] [max_uses]
```

### المعاملات:

| المعامل | الوصف | مطلوب | الافتراضي |
|---------|-------|-------|----------|
| `type` | نوع القسيمة (`users`/`days`) | ✅ | - |
| `amount` | الكمية | ✅ | - |
| `target` | البوت المستهدف أو `all` | ❌ | `all` |
| `max_uses` | عدد الاستخدامات | ❌ | `1` |

---

### أمثلة:

#### 1. قسيمة لجميع البوتات (استخدام واحد)

```bash
/generate_voucher users 100
```

**النتيجة**:
- نوع: `users`
- كمية: 100 مستخدم
- الهدف: `all` (جميع البوتات)
- الاستخدامات: 1

---

#### 2. قسيمة لبوت معين

```bash
/generate_voucher users 50 test1
```

**النتيجة**:
- نوع: `users`
- كمية: 50 مستخدم
- الهدف: بوت `test1` فقط
- الاستخدامات: 1

---

#### 3. قسيمة متعددة الاستخدام

```bash
/generate_voucher users 200 all 5
```

**النتيجة**:
- نوع: `users`
- كمية: 200 مستخدم
- الهدف: `all`
- الاستخدامات: 5 (يمكن تفعيلها 5 مرات)

---

### كود الإنشاء:

```php
case '/generate_voucher': {
    // التحقق من المدخلات
    if (count($args) < 2) {
        tg_send($CONTROL_TOKEN, $chat_id, "الاستخدام...");
        break;
    }
    
    // استخراج المعاملات
    $type = strtolower(trim($args[0]));
    $amount = (int)$args[1];
    $target = $args[2] ?? 'all';
    $max_uses = isset($args[3]) ? (int)$args[3] : 1;
    
    // التحقق من صحة النوع
    if (!in_array($type, ['users', 'days'])) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ نوع خاطئ!");
        break;
    }
    
    // التحقق من القيم الموجبة
    if ($amount <= 0 || $max_uses <= 0) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ القيمة يجب أن تكون أكبر من صفر!");
        break;
    }
    
    // توليد الكود
    $code_gen = 'V' . strtoupper(bin2hex(random_bytes(5)));
    $target_label = ($target === 'all') ? null : $target;
    
    // الإدراج في قاعدة البيانات
    $stmt = $db->prepare("INSERT INTO vouchers(code, voucher_type, amount, target_label, max_uses, created_by_user_id) VALUES(?,?,?,?,?,?)");
    $result = $stmt->execute([$code_gen, $type, $amount, $target_label, $max_uses, $from_id]);
    
    // إرسال النتيجة
    $out = "✅ <b>تم إنشاء القسيمة</b>\n\n";
    $out .= "🎫 الكود: <code>{$code_gen}</code>\n";
    $out .= "📦 النوع: " . ($type === 'users' ? '👥 مستخدمين' : '📅 أيام') . "\n";
    $out .= "💰 القيمة: {$amount}\n";
    $out .= "🎯 الهدف: " . ($target === 'all' ? 'جميع البوتات' : "<b>{$target}</b>") . "\n";
    $out .= "🔢 الاستخدامات: {$max_uses}\n\n";
    $out .= "💡 للتفعيل: <code>/redeem {$code_gen}" . ($target === 'all' ? ' bot_label' : '') . "</code>";
    
    tg_send($CONTROL_TOKEN, $chat_id, $out);
}
```

---

### مثال على الكود المُنشأ:

```
VA3F2B9C1D4E
VF7A8D2E5B3C
V12AB34CD56EF
```

**التنسيق**: حرف `V` + 10 أحرف هيكسا (Hex) كبيرة

---

## ✅ تفعيل القسائم

### في بوت التحكم المركزي

**الأمر**: `/redeem <code> [bot_label]`

**الملف**: `control_bot.php` السطر 951

**من يستطيع**: المالك فقط

---

### في البوتات الفرعية

**الأمر**: `/redeem <code>`

**الملف**: `sub_bot.php` السطر 456

**من يستطيع**: المدراء الفرعيون بصلاحية (3)

---

### الفرق بين النظامين:

| الميزة | بوت التحكم | البوتات الفرعية |
|--------|-----------|-----------------|
| تحديد البوت | يدوياً (إلزامي إذا `target=all`) | تلقائياً (البوت الحالي) |
| الصلاحية | المالك فقط | مدير بصلاحية (3) |
| التسجيل | ✅ `voucher_redemptions` | ✅ `voucher_redemptions` |

---

### خطوات التفعيل (مشتركة):

```
1. التحقق من وجود الكود في `vouchers`
    ↓
2. فحص: هل تم استهلاكه؟ (current_uses >= max_uses)
    ↓ (نعم → رفض)
    ↓ (لا)
    ↓
3. تحديد البوت المستهدف:
   - إذا target_label محدد → استخدامه
   - وإلا → يجب تحديد bot_label يدوياً
    ↓
4. التحقق من وجود البوت في sub_bots
    ↓
5. تطبيق القسيمة حسب النوع:
   - users: UPDATE sub_bots SET max_users = max_users + amount
   - days: (غير مفعّل)
    ↓
6. زيادة عداد الاستخدام:
   UPDATE vouchers SET current_uses = current_uses + 1
    ↓
7. التسجيل في voucher_redemptions
    ↓
8. إرسال رسالة نجاح
```

---

### كود التفعيل (بوت التحكم):

```php
case '/redeem': {
    $code_input = strtoupper($args[0]);
    
    // جلب القسيمة
    $stmt = $db->prepare("SELECT * FROM vouchers WHERE code=?");
    $stmt->execute([$code_input]);
    $voucher = $stmt->fetch();
    
    if (!$voucher) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ القسيمة غير موجودة");
        break;
    }
    
    // فحص الاستهلاك
    if ($voucher['current_uses'] >= $voucher['max_uses']) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ تم استهلاك القسيمة ({$voucher['current_uses']}/{$voucher['max_uses']})");
        break;
    }
    
    // تحديد البوت
    if ($voucher['target_label']) {
        $label = $voucher['target_label'];
    } else {
        if (!isset($args[1])) {
            tg_send($CONTROL_TOKEN, $chat_id, "⚠️ حدد البوت:\n<code>/redeem {$code_input} bot_label</code>");
            break;
        }
        $label = $args[1];
    }
    
    // التحقق من وجود البوت
    $stmt = $db->prepare("SELECT id FROM sub_bots WHERE label=?");
    $stmt->execute([$label]);
    if (!$stmt->fetch()) {
        tg_send($CONTROL_TOKEN, $chat_id, "❌ البوت <b>{$label}</b> غير موجود");
        break;
    }
    
    // تطبيق القسيمة
    if ($voucher['voucher_type'] === 'users') {
        $db->prepare("UPDATE sub_bots SET max_users = max_users + ? WHERE label=?")
           ->execute([$voucher['amount'], $label]);
    } else {
        tg_send($CONTROL_TOKEN, $chat_id, "⚠️ نوع 'days' غير مدعوم حالياً");
        break;
    }
    
    // زيادة عداد الاستخدام
    $db->prepare("UPDATE vouchers SET current_uses = current_uses + 1 WHERE code=?")
       ->execute([$code_input]);
    
    // إرسال النتيجة
    tg_send($CONTROL_TOKEN, $chat_id, "✅ <b>تم تفعيل القسيمة</b>\n\n🤖 البوت: <b>{$label}</b>\n💰 تمت إضافة: {$voucher['amount']} مستخدم");
}
```

---

### كود التفعيل (البوتات الفرعية):

```php
case '/redeem': {
    // التحقق من الصلاحيات
    if (!has_permission($user_id, 3, $token)) {
        tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية تعبئة القسائم");
        exit;
    }
    
    $code = strtoupper($arg);
    
    // جلب القسيمة
    $v = $db->prepare("SELECT * FROM vouchers WHERE code=? LIMIT 1");
    $v->execute([$code]);
    $voucher = $v->fetch();
    
    if (!$voucher) {
        tg_send($token, $chat_id, "❌ القسيمة غير موجودة");
        exit;
    }
    
    // فحص الاستهلاك
    if ($voucher['current_uses'] >= $voucher['max_uses']) {
        tg_send($token, $chat_id, "❌ تم استهلاك القسيمة");
        exit;
    }
    
    // تطبيق القسيمة (البوت الحالي تلقائياً)
    if ($voucher['voucher_type'] === 'users') {
        $db->prepare("UPDATE sub_bots SET max_users = max_users + ? WHERE bot_token=?")
           ->execute([$voucher['amount'], $token]);
    }
    
    // زيادة عداد الاستخدام
    $db->prepare("UPDATE vouchers SET current_uses = current_uses + 1 WHERE code=?")
       ->execute([$code]);
    
    // التسجيل
    $db->prepare("INSERT INTO voucher_redemptions(voucher_code, bot_token, redeemed_by, amount_added) VALUES(?,?,?,?)")
       ->execute([$code, $token, $user_id, $voucher['amount']]);
    
    // إرسال النتيجة
    tg_send($token, $chat_id, "✅ <b>تم تفعيل القسيمة</b>\n\n💰 تمت إضافة: {$voucher['amount']} مستخدم");
}
```

---

## 📊 عرض القسائم

### في بوت التحكم

#### `/vouchers` - عرض آخر 50 قسيمة

```php
case '/vouchers': {
    $vouchers = $db->query("SELECT * FROM vouchers ORDER BY id DESC LIMIT 50")->fetchAll();
    
    if (!$vouchers) {
        tg_send($CONTROL_TOKEN, $chat_id, "لا توجد قسائم");
        break;
    }
    
    $out = "🎫 <b>القسائم (آخر 50):</b>\n\n";
    foreach ($vouchers as $v) {
        $status = ($v['current_uses'] >= $v['max_uses']) ? '❌' : '✅';
        $target = $v['target_label'] ? $v['target_label'] : 'الكل';
        $out .= "{$status} <code>{$v['code']}</code>\n";
        $out .= "  └ {$v['voucher_type']} | {$v['amount']} | {$target} | {$v['current_uses']}/{$v['max_uses']}\n\n";
    }
    tg_send($CONTROL_TOKEN, $chat_id, $out);
}
```

**المخرج**:
```
🎫 القسائم (آخر 50):

✅ VA3F2B9C1D4E
  └ users | 100 | الكل | 0/1

✅ VF7A8D2E5B3C
  └ users | 50 | test1 | 1/5

❌ V12AB34CD56EF
  └ users | 200 | الكل | 3/3
```

---

#### `/voucher_search <code>` - البحث عن قسيمة

**الملف**: `control_bot.php` السطر 1020

**الوظيفة**: عرض تفاصيل قسيمة معينة

---

### في البوتات الفرعية

#### `/vouchers_log` - سجل القسائم المستخدمة

```php
case '/vouchers_log': {
    if (!has_permission($user_id, 3, $token)) {
        tg_send($token, $chat_id, "⛔️ ليس لديك صلاحية عرض سجل القسائم");
        exit;
    }
    
    $logs = $db->prepare("SELECT * FROM voucher_redemptions WHERE bot_token=? ORDER BY id DESC LIMIT 20");
    $logs->execute([$token]);
    $list = $logs->fetchAll();
    
    if (!$list) {
        tg_send($token, $chat_id, "لا يوجد سجل قسائم");
        exit;
    }
    
    $out = "📜 <b>سجل القسائم:</b>\n\n";
    $num = 1;
    foreach ($list as $l) {
        $out .= "{$num}️⃣ <code>{$l['voucher_code']}</code> | ✅ مستخدمة\n";
        $out .= "   📅 {$l['redeemed_at']} | +{$l['amount_added']} مستخدم\n\n";
        $num++;
    }
    
    tg_send($token, $chat_id, $out);
}
```

---

## 🔗 ربط القسائم بالمستخدمين/البوتات

### الوضع الحالي:

**القسائم مرتبطة بـ**:
- ✅ البوتات (`target_label` أو يدوياً عند التفعيل)
- ✅ من فعّلها (`redeemed_by` في `voucher_redemptions`)

**القسائم غير مرتبطة بـ**:
- ❌ المجموعات (لا يوجد دعم حالياً)
- ❌ مستخدمين محددين (غير مطبق)

---

### هل يوجد دعم للمجموعات؟

**الإجابة**: ❌ **لا، حالياً**

القسائم تستهدف فقط:
- بوت واحد (`target_label = 'test1'`)
- جميع البوتات (`target_label = NULL`)

**لا يمكن إنشاء قسيمة**:
- لمجموعة معينة (مثل "36k")
- لمستخدمين محددين

---

## 🎯 حالات الاستخدام (Use Cases)

### 1. بيع قسائم

```
المالك يُنشئ: /generate_voucher users 100 all 1
الكود: VA3F2B9C1D4E
العميل يشتري القسيمة
المدير يفعّلها: /redeem VA3F2B9C1D4E mybotlabel
```

---

### 2. مكافأة مدير

```
المالك يُنشئ: /generate_voucher users 50 test1 5
الكود: VF7A8D2E5B3C
المدير يفعّلها في البوت: /redeem VF7A8D2E5B3C
(يمكنه تفعيلها 5 مرات = 250 مستخدم إجمالاً)
```

---

### 3. عرض ترويجي

```
المالك يُنشئ: /generate_voucher users 200 all 100
الكود: V12AB34CD56EF
توزيع الكود على 100 مدير
كل مدير يفعّله في بوته
```

---

## ⚠️ المشاكل والقيود الحالية

### 1. لا يوجد تحقق من `target_label`

**المشكلة**: يمكن إنشاء قسيمة لبوت غير موجود

```bash
/generate_voucher users 100 nonexistent_bot
```

**النتيجة**: تُنشأ القسيمة لكن لا يمكن تفعيلها

**الحل المقترح**: التحقق من وجود البوت قبل الإنشاء

---

### 2. نوع `days` غير مفعّل

**المشكلة**: يمكن إنشاء قسيمة `days` لكن لا يمكن تفعيلها

```bash
/generate_voucher days 30 test1
```

**عند التفعيل**:
```
⚠️ نوع 'days' غير مدعوم حالياً
```

**الحل المقترح**: إزالة `days` من القائمة أو تطبيقه

---

### 3. لا يوجد تاريخ انتهاء

**المشكلة**: القسائم لا تنتهي صلاحيتها أبداً

**الحل المقترح**: إضافة حقل `expires_at` في `vouchers`

---

### 4. لا يوجد ربط بالمجموعات

**المشكلة**: لا يمكن إنشاء قسيمة لمجموعة معينة

**مثال مطلوب**:
```bash
/generate_voucher users 100 group:36k
```

**الحل المقترح**: توسعة `target_label` أو إضافة حقل `target_group_id`

---

## 💡 توصيات للتطوير

### 1. إضافة تواريخ انتهاء

```sql
ALTER TABLE vouchers 
ADD COLUMN `expires_at` timestamp NULL;
```

**الاستخدام**:
```bash
/generate_voucher users 100 all 1 30d
```
(تنتهي بعد 30 يوم)

---

### 2. دعم المجموعات

```sql
ALTER TABLE vouchers 
ADD COLUMN `target_group_id` int(11) DEFAULT NULL,
ADD FOREIGN KEY (`target_group_id`) REFERENCES `account_groups`(`id`) ON DELETE SET NULL;
```

**الاستخدام**:
```bash
/generate_voucher_group users 50 36k
```

---

### 3. تفعيل نوع `days`

**المنطق المقترح**:
```php
if ($voucher['voucher_type'] === 'days') {
    // إضافة أيام لحقل جديد `premium_until` في sub_bots
    $db->prepare("UPDATE sub_bots SET premium_until = DATE_ADD(COALESCE(premium_until, NOW()), INTERVAL ? DAY) WHERE label=?")
       ->execute([$voucher['amount'], $label]);
}
```

---

### 4. قسائم للمستخدمين

**السيناريو**: قسيمة تمنح مستخدم معين حداً خاصاً

```sql
CREATE TABLE `user_vouchers` (
  `id` bigint(20) PRIMARY KEY AUTO_INCREMENT,
  `voucher_code` varchar(32) NOT NULL,
  `user_id` bigint(20) NOT NULL,
  `benefit_type` enum('unlimited_codes','premium_days') NOT NULL,
  `benefit_value` int(11) DEFAULT NULL,
  `expires_at` timestamp NULL,
  UNIQUE KEY (`voucher_code`, `user_id`)
);
```

---

## 📊 إحصائيات القسائم (مقترحة)

### أوامر مفيدة:

```bash
/voucher_stats          # عرض إحصائيات عامة
/voucher_most_used      # أكثر القسائم استخداماً
/voucher_by_bot test1   # قسائم بوت معين
```

---

## 🎯 الخلاصة

**نقاط القوة**:
- ✅ نظام مرن (متعدد الأنواع والاستخدامات)
- ✅ تسجيل كامل في `voucher_redemptions`
- ✅ دعم البوتات الفرعية (صلاحية 3)
- ✅ آمن (UNIQUE على `code`)

**نقاط الضعف**:
- ⚠️ نوع `days` غير مفعّل
- ⚠️ لا يوجد تاريخ انتهاء
- ⚠️ لا دعم للمجموعات
- ⚠️ لا تحقق من `target_label` عند الإنشاء

**التوصية**: النظام يعمل بشكل جيد لحالة الاستخدام الحالية (زيادة المستخدمين)، لكن يحتاج لتوسعات إذا أُريد دعم المجموعات أو أنواع أخرى من القسائم.
