# تقرير التوصيات للتطوير (Development Recommendations)

## 📋 نظرة عامة

هذا التقرير يقدم **توصيات عامة** لتحسين المشروع بدون الدخول في التنفيذ الفعلي.

---

## 1️⃣ نظام تقسيم الرسائل الطويلة (Message Splitting)

### المشكلة

الأوامر `/list` و `/allusers` تفشل عند تجاوز 4096 حرف.

### الحل المقترح

#### الخيار A: دالة مساعدة للتقسيم

```php
function tg_send_long($token, $chat_id, $text, $chunk_size = 3500) {
    if (strlen($text) <= 4000) {
        return tg_send($token, $chat_id, $text);
    }
    
    $chunks = str_split($text, $chunk_size);
    foreach ($chunks as $i => $chunk) {
        tg_send($token, $chat_id, "صفحة " . ($i+1) . ":\n\n" . $chunk);
        usleep(200000);
    }
}
```

#### الخيار B: Pagination

```php
/list [page]
/allusers [page]
```

مع أزرار Inline للتنقل:
```
[← السابق] [التالي →]
```

---

## 2️⃣ نظام Queue للبث (Broadcast Queue)

### المشكلة

البث يتم بشكل متزامن، مما يؤخر الاستجابة.

### الحل المقترح

#### استخدام جدول `broadcast_jobs`

**الخطوة 1**: إنشاء مهمة بث
```php
case '/broadcast': {
    $job_key = md5($token . $payload . time());
    $db->prepare("INSERT INTO broadcast_jobs(job_key, initiator_user_id, target_type, target_value, message_text, status) VALUES(?,?,?,?,?,'pending')")
        ->execute([$job_key, $user_id, 'bot', $bot_label, $payload]);
    
    tg_send($CONTROL_TOKEN, $chat_id, "✅ تم إنشاء مهمة البث\n\nسيتم إرسال تنبيه عند الانتهاء");
}
```

**الخطوة 2**: Worker Script (Cron)
```php
// broadcast_worker.php
while (true) {
    $job = $db->query("SELECT * FROM broadcast_jobs WHERE status='pending' LIMIT 1")->fetch();
    if (!$job) {
        sleep(5);
        continue;
    }
    
    // تحديث الحالة
    $db->prepare("UPDATE broadcast_jobs SET status='running' WHERE id=?")->execute([$job['id']]);
    
    // تنفيذ البث
    // ...
    
    // تحديث النتيجة
    $db->prepare("UPDATE broadcast_jobs SET status='completed', completed_at=NOW() WHERE id=?")->execute([$job['id']]);
    
    // إرسال تنبيه للمالك
    tg_send($CONTROL_TOKEN, $job['initiator_user_id'], "✅ اكتمل البث");
}
```

**Cron Job**:
```bash
*/1 * * * * php /path/to/broadcast_worker.php
```

---

## 3️⃣ توسعة نظام الردود (Response System)

### الوضع الحالي

الرسائل في جدول `message_templates` و `limit_messages`.

### التحسين المقترح

#### نظام موحد وديناميكي

```sql
CREATE TABLE `response_templates` (
  `id` int(11) PRIMARY KEY AUTO_INCREMENT,
  `key` varchar(64) UNIQUE NOT NULL,
  `text` text NOT NULL,
  `variables` text,  -- JSON array of variables
  `category` enum('general','limits','vouchers','groups','broadcast') NOT NULL,
  `language` varchar(2) DEFAULT 'ar',
  `updated_at` timestamp DEFAULT current_timestamp() ON UPDATE current_timestamp()
);
```

#### أوامر الإدارة

```bash
/response_list [category]         # عرض القوالب
/response_edit <key> <text>       # تعديل
/response_preview <key>           # معاينة
```

#### ميزات إضافية

- **متعدد اللغات**: حقل `language`
- **متغيرات ديناميكية**: JSON في `variables`
- **تصنيف**: للتنظيم

---

## 4️⃣ تقسيم الأوامر إلى Modules

### الوضع الحالي

`control_bot.php` يحتوي على **Switch Statement ضخم** (~3000 سطر).

### التحسين المقترح

#### بنية المجلدات

```
project/
├── lib/
│   ├── telegram.php
│   ├── limits.php
│   └── ...
├── modules/          # ← جديد
│   ├── bots.php      # أوامر البوتات
│   ├── users.php     # أوامر المستخدمين
│   ├── accounts.php  # أوامر الحسابات
│   ├── groups.php    # أوامر المجموعات
│   ├── vouchers.php  # أوامر القسائم
│   ├── limits.php    # أوامر الحدود
│   ├── broadcast.php # أوامر البث
│   └── stats.php     # أوامر الإحصائيات
├── control_bot.php
└── sub_bot.php
```

#### مثال: `modules/bots.php`

```php
<?php
function handle_bot_commands($cmd, $args, $db, $chat_id, $from_id, $CONTROL_TOKEN) {
    switch ($cmd) {
        case '/addbot':
            // منطق /addbot
            break;
        case '/list':
            // منطق /list
            break;
        // ... باقي أوامر البوتات
    }
}
```

#### `control_bot.php` يصبح

```php
<?php
require __DIR__ . '/modules/bots.php';
require __DIR__ . '/modules/users.php';
// ...

// التوجيه
if (in_array($cmd, ['/addbot', '/list', '/pause', '/resume', ...])) {
    handle_bot_commands($cmd, $args, $db, $chat_id, $from_id, $CONTROL_TOKEN);
} elseif (in_array($cmd, ['/users', '/allusers', '/ban', ...])) {
    handle_user_commands($cmd, $args, $db, $chat_id, $from_id, $CONTROL_TOKEN);
}
// ...
```

---

## 5️⃣ نظام Logging شامل

### الوضع الحالي

`error_log()` فقط، لا يوجد نظام موحد.

### التحسين المقترح

#### استخدام جدول `system_logs`

```php
function log_event($level, $component, $message, $user_id = null, $data = null) {
    global $db;
    $stmt = $db->prepare("INSERT INTO system_logs(log_level, component, message, user_id, data) VALUES(?,?,?,?,?)");
    $stmt->execute([$level, $component, $message, $user_id, json_encode($data)]);
}
```

#### استخدام

```php
// في البث
log_event('INFO', 'broadcast', 'Broadcast started', $user_id, ['bot' => $label, 'recipients' => count($users)]);

// في الأخطاء
log_event('ERROR', 'voucher', 'Redemption failed', $user_id, ['code' => $code, 'error' => $e->getMessage()]);
```

#### أوامر المراقبة

```bash
/logs [level] [component] [limit]  # عرض السجلات
/logs ERROR broadcast 50
```

---

## 6️⃣ نظام Caching للبيانات المتكررة

### المشكلة

كل طلب يستعلم DB للحدود والإعدادات.

### الحل المقترح

#### استخدام APCu أو Redis

```php
function get_limits_cached($bot_label, $account_id) {
    $cache_key = "limits:{$bot_label}:{$account_id}";
    
    if (apcu_exists($cache_key)) {
        return apcu_fetch($cache_key);
    }
    
    $limits = get_effective_limits($db, $bot_label, $account_id);
    apcu_store($cache_key, $limits, 300); // 5 دقائق
    
    return $limits;
}
```

#### إبطال Cache عند التحديث

```php
function update_bot_limits($bot_label, $mode, $per_day) {
    global $db;
    $db->prepare("UPDATE limits_bot SET mode=?, per_day=? WHERE bot_label=?")->execute(...);
    
    // إبطال Cache
    apcu_clear_cache();
    // أو بشكل محدد
    apcu_delete("limits:{$bot_label}:*");
}
```

---

## 7️⃣ API Endpoints للتكامل الخارجي

### الفكرة

توفير API لإدارة النظام خارجياً (لوحة تحكم ويب).

### التصميم المقترح

#### ملف جديد: `api.php`

```php
<?php
require __DIR__ . '/lib/telegram.php';

$api_key = $_GET['api_key'] ?? '';
if ($api_key !== 'SECRET_KEY') {
    http_response_code(401);
    exit(json_encode(['error' => 'Unauthorized']));
}

$action = $_GET['action'] ?? '';

switch ($action) {
    case 'list_bots':
        $bots = $db->query("SELECT label, is_paused, max_users FROM sub_bots")->fetchAll();
        echo json_encode($bots);
        break;
    
    case 'add_user':
        $bot_token = $_POST['bot_token'];
        $user_id = $_POST['user_id'];
        $db->prepare("INSERT IGNORE INTO allowed_users(bot_token, user_id) VALUES(?,?)")->execute([$bot_token, $user_id]);
        echo json_encode(['success' => true]);
        break;
    
    // ... باقي Endpoints
}
```

#### الاستخدام

```bash
curl "https://fairouz.shop/api.php?api_key=SECRET&action=list_bots"
```

---

## 8️⃣ لوحة تحكم ويب (Web Dashboard)

### الفكرة

واجهة ويب لإدارة النظام بدلاً من الأوامر النصية فقط.

### المميزات المقترحة

- **إحصائيات مباشرة** (عدد المستخدمين، الطلبات اليومية، ...)
- **إدارة البوتات** (إضافة/تعديل/حذف)
- **إدارة المستخدمين** (جداول قابلة للبحث)
- **إدارة المجموعات** (Drag & Drop للربط)
- **نظام البث** (محرر نصوص + معاينة)
- **تقارير ورسوم بيانية** (Charts.js)

### التقنيات المقترحة

- **Frontend**: React.js أو Vue.js
- **Backend**: API.php (كما في النقطة السابقة)
- **UI Framework**: TailwindCSS + shadcn/ui
- **Charts**: Chart.js أو ApexCharts

---

## 9️⃣ نظام Backup تلقائي

### الفكرة

نسخ احتياطي تلقائي لقاعدة البيانات.

### التصميم المقترح

#### Script: `backup.php`

```php
<?php
$cfg = require __DIR__ . '/.env.php';
$filename = 'backup_' . date('Y-m-d_H-i-s') . '.sql';
$command = "mysqldump -u{$cfg['DB_USER']} -p{$cfg['DB_PASS']} {$cfg['DB_NAME']} > backups/{$filename}";
exec($command);

// حذف Backups أقدم من 30 يوم
foreach (glob('backups/*.sql') as $file) {
    if (time() - filemtime($file) > 30 * 86400) {
        unlink($file);
    }
}
```

#### Cron Job

```bash
0 3 * * * php /path/to/backup.php
```

#### أمر يدوي

```bash
/backup_db  # يُنفذ Backup فوري
```

---

## 🔟 نظام Webhooks للأحداث

### الفكرة

إرسال إشعارات لـ URL خارجي عند أحداث معينة.

### الأحداث المقترحة

- `user.registered` - مستخدم جديد
- `broadcast.completed` - انتهاء بث
- `voucher.redeemed` - تفعيل قسيمة
- `limit.exceeded` - تجاوز حد
- `bot.added` - إضافة بوت

### التصميم

#### جدول `webhooks`

```sql
CREATE TABLE `webhooks` (
  `id` int(11) PRIMARY KEY AUTO_INCREMENT,
  `event` varchar(64) NOT NULL,
  `url` varchar(255) NOT NULL,
  `secret` varchar(64),
  `active` tinyint(1) DEFAULT 1
);
```

#### إرسال Webhook

```php
function trigger_webhook($event, $data) {
    global $db;
    $webhooks = $db->prepare("SELECT * FROM webhooks WHERE event=? AND active=1")->execute([$event])->fetchAll();
    
    foreach ($webhooks as $wh) {
        $payload = json_encode([
            'event' => $event,
            'data' => $data,
            'timestamp' => time()
        ]);
        
        $ch = curl_init($wh['url']);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'X-Webhook-Secret: ' . $wh['secret']
        ]);
        curl_exec($ch);
    }
}
```

---

## 🎯 ملخص الأولويات

| # | التوصية | الأولوية | التأثير |
|---|---------|----------|----------|
| 1 | تقسيم الرسائل الطويلة | 🔴 عالية | 🔴 عالي |
| 2 | DISTINCT في البث | 🔴 عالية | 🟡 متوسط |
| 3 | Transactions | 🟡 متوسطة | 🟡 متوسط |
| 4 | Queue للبث | 🟡 متوسطة | 🟢 منخفض |
| 5 | تقسيم Modules | 🟢 منخفضة | 🟡 متوسط |
| 6 | نظام Logging | 🟡 متوسطة | 🟡 متوسط |
| 7 | Caching | 🟢 منخفضة | 🟢 منخفض |
| 8 | API Endpoints | 🟢 منخفضة | 🟢 منخفض |
| 9 | Web Dashboard | 🟢 منخفضة | 🟢 منخفض |
| 10 | Backup تلقائي | 🟡 متوسطة | 🟡 متوسط |
| 11 | Webhooks | 🟢 منخفضة | 🟢 منخفض |

---

## 📝 خطة التطوير المقترحة

### المرحلة 1: الإصلاحات الحرجة (أسبوع 1)

1. ✅ إضافة `DISTINCT` في البث
2. ✅ تطبيق تقسيم الرسائل في `/list` و `/allusers`
3. ✅ إضافة Transactions في القسائم

### المرحلة 2: التحسينات الأساسية (أسبوع 2-3)

4. ✅ نظام Queue للبث
5. ✅ نظام Logging
6. ✅ إضافة `/groups_list`
7. ✅ تحسين رسائل الأخطاء

### المرحلة 3: إعادة الهيكلة (أسبوع 4-6)

8. ✅ تقسيم الأوامر إلى Modules
9. ✅ توحيد نظام الردود
10. ✅ إضافة Caching

### المرحلة 4: الميزات الإضافية (المستقبل)

11. ⏳ API Endpoints
12. ⏳ Web Dashboard
13. ⏳ Webhooks
14. ⏳ تقارير متقدمة

---

## 🎯 الخلاصة

النظام الحالي **قوي ويعمل بشكل جيد**. التوصيات الواردة هنا هي **تحسينات** وليست إصلاحات حرجة. يمكن تنفيذها تدريجياً حسب الأولوية والوقت المتاح.

**أهم 3 توصيات للبدء فوراً**:
1. تقسيم الرسائل الطويلة
2. إضافة DISTINCT في البث
3. استخدام Transactions
