<?php
/**
 * ===============================================
 * سكربت التحقق من الترحيل - Verification Script
 * ===============================================
 * 
 * يتحقق من سلامة البيانات بعد الترحيل
 */

require_once __DIR__ . '/../config/migration_config.ini';

define('ROOT_DIR', dirname(__DIR__));
define('CONFIG_FILE', ROOT_DIR . '/config/migration_config.ini');

class MigrationVerifier {
    private $sourceDB;
    private $targetDB;
    private $config;
    private $errors = [];
    private $warnings = [];
    
    public function __construct($configFile) {
        $this->config = parse_ini_file($configFile, true);
        $this->connectDatabases();
    }
    
    private function connectDatabases() {
        try {
            $src = $this->config['source_database'];
            $this->sourceDB = new PDO(
                "mysql:host={$src['host']};port={$src['port']};dbname={$src['database']};charset={$src['charset']}",
                $src['username'],
                $src['password'],
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            $tgt = $this->config['target_database'];
            $this->targetDB = new PDO(
                "mysql:host={$tgt['host']};port={$tgt['port']};dbname={$tgt['database']};charset={$tgt['charset']}",
                $tgt['username'],
                $tgt['password'],
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            echo "✅ اتصال ناجح بالقاعدتين\n\n";
            
        } catch (PDOException $e) {
            die("❌ فشل الاتصال: " . $e->getMessage() . "\n");
        }
    }
    
    public function verify() {
        echo "╔══════════════════════════════════════════════════════════╗\n";
        echo "║                                                          ║\n";
        echo "║          🔍 التحقق من الترحيل 🔍                       ║\n";
        echo "║                                                          ║\n";
        echo "╚══════════════════════════════════════════════════════════╝\n\n";
        
        // 1. مقارنة عدد السجلات
        $this->verifyRecordCounts();
        
        // 2. التحقق من العلاقات
        $this->verifyRelationships();
        
        // 3. التحقق من البيانات اليتيمة
        $this->verifyOrphans();
        
        // 4. التحقق من Foreign Keys
        $this->verifyForeignKeys();
        
        // 5. التحقق من الفهارس
        $this->verifyIndexes();
        
        // التقرير النهائي
        $this->printReport();
    }
    
    private function verifyRecordCounts() {
        echo "═══════════════════════════════════════════════════════════\n";
        echo "1️⃣ مقارنة عدد السجلات\n";
        echo "═══════════════════════════════════════════════════════════\n\n";
        
        $tables = ['sub_bots', 'steam_accounts', 'allowed_users', 'message_templates'];
        
        foreach ($tables as $table) {
            $srcCount = $this->sourceDB->query("SELECT COUNT(*) FROM $table")->fetchColumn();
            $tgtCount = $this->targetDB->query("SELECT COUNT(*) FROM $table")->fetchColumn();
            
            $status = ($srcCount == $tgtCount) ? '✅' : '❌';
            $diff = $tgtCount - $srcCount;
            $diffStr = $diff > 0 ? "+$diff" : ($diff < 0 ? "$diff" : "0");
            
            printf("%-20s: %5d → %5d (%s) %s\n", $table, $srcCount, $tgtCount, $diffStr, $status);
            
            if ($srcCount != $tgtCount) {
                $this->errors[] = "$table: عدد السجلات غير متطابق ($srcCount → $tgtCount)";
            }
        }
        
        echo "\n";
    }
    
    private function verifyRelationships() {
        echo "═══════════════════════════════════════════════════════════\n";
        echo "2️⃣ التحقق من العلاقات\n";
        echo "═══════════════════════════════════════════════════════════\n\n";
        
        // عدد الحسابات vs group_accounts
        $accounts = $this->targetDB->query("SELECT COUNT(*) FROM steam_accounts")->fetchColumn();
        $linked = $this->targetDB->query("SELECT COUNT(*) FROM group_accounts WHERE group_id = 1")->fetchColumn();
        
        $status = ($accounts == $linked) ? '✅' : '❌';
        printf("steam_accounts        : %5d\n", $accounts);
        printf("group_accounts (g=1)  : %5d %s\n", $linked, $status);
        
        if ($accounts != $linked) {
            $this->errors[] = "عدد الحسابات ($accounts) لا يساوي الحسابات المربوطة ($linked)";
        }
        
        echo "\n";
        
        // عدد البوتات vs bot_groups
        $bots = $this->targetDB->query("SELECT COUNT(*) FROM sub_bots")->fetchColumn();
        $linkedBots = $this->targetDB->query("SELECT COUNT(*) FROM bot_groups WHERE group_id = 1")->fetchColumn();
        
        $status = ($bots == $linkedBots) ? '✅' : '❌';
        printf("sub_bots              : %5d\n", $bots);
        printf("bot_groups (g=1)      : %5d %s\n", $linkedBots, $status);
        
        if ($bots != $linkedBots) {
            $this->errors[] = "عدد البوتات ($bots) لا يساوي البوتات المربوطة ($linkedBots)";
        }
        
        echo "\n";
    }
    
    private function verifyOrphans() {
        echo "═══════════════════════════════════════════════════════════\n";
        echo "3️⃣ البحث عن البيانات اليتيمة (Orphans)\n";
        echo "═══════════════════════════════════════════════════════════\n\n";
        
        // حسابات بدون مجموعة
        $orphanedAccounts = $this->targetDB->query("
            SELECT COUNT(*) FROM steam_accounts sa
            LEFT JOIN group_accounts ga ON ga.account_id = sa.id
            WHERE ga.account_id IS NULL
        ")->fetchColumn();
        
        $status = ($orphanedAccounts == 0) ? '✅' : '❌';
        printf("حسابات يتيمة (بدون مجموعة)  : %5d %s\n", $orphanedAccounts, $status);
        
        if ($orphanedAccounts > 0) {
            $this->errors[] = "يوجد $orphanedAccounts حساب بدون مجموعة";
        }
        
        // بوتات بدون مجموعة
        $orphanedBots = $this->targetDB->query("
            SELECT COUNT(*) FROM sub_bots sb
            LEFT JOIN bot_groups bg ON bg.bot_id = sb.id
            WHERE bg.bot_id IS NULL
        ")->fetchColumn();
        
        $status = ($orphanedBots == 0) ? '✅' : '❌';
        printf("بوتات يتيمة (بدون مجموعة)    : %5d %s\n", $orphanedBots, $status);
        
        if ($orphanedBots > 0) {
            $this->errors[] = "يوجد $orphanedBots بوت بدون مجموعة";
        }
        
        // allowed_users بدون بوت موجود
        $orphanedUsers = $this->targetDB->query("
            SELECT COUNT(*) FROM allowed_users au
            LEFT JOIN sub_bots sb ON sb.bot_token = au.bot_token
            WHERE sb.id IS NULL
        ")->fetchColumn();
        
        $status = ($orphanedUsers == 0) ? '✅' : '⚠️';
        printf("مستخدمون بدون بوت موجود      : %5d %s\n", $orphanedUsers, $status);
        
        if ($orphanedUsers > 0) {
            $this->warnings[] = "يوجد $orphanedUsers مستخدم مربوط ببوت غير موجود";
        }
        
        echo "\n";
    }
    
    private function verifyForeignKeys() {
        echo "═══════════════════════════════════════════════════════════\n";
        echo "4️⃣ التحقق من Foreign Keys\n";
        echo "═══════════════════════════════════════════════════════════\n\n";
        
        try {
            $this->targetDB->exec("SET FOREIGN_KEY_CHECKS=1");
            echo "✅ جميع Foreign Keys سليمة\n\n";
        } catch (PDOException $e) {
            echo "❌ خطأ في Foreign Keys: " . $e->getMessage() . "\n\n";
            $this->errors[] = "FK Error: " . $e->getMessage();
        }
    }
    
    private function verifyIndexes() {
        echo "═══════════════════════════════════════════════════════════\n";
        echo "5️⃣ التحقق من الفهارس\n";
        echo "═══════════════════════════════════════════════════════════\n\n";
        
        $tables = ['sub_bots', 'steam_accounts', 'allowed_users', 'group_accounts', 'bot_groups'];
        
        foreach ($tables as $table) {
            try {
                $indexes = $this->targetDB->query("SHOW INDEXES FROM $table")->fetchAll();
                $indexCount = count($indexes);
                echo "✅ $table: $indexCount فهرس\n";
            } catch (PDOException $e) {
                echo "❌ $table: خطأ في الفهارس\n";
                $this->errors[] = "$table: خطأ في الفهارس";
            }
        }
        
        echo "\n";
    }
    
    private function printReport() {
        echo "═══════════════════════════════════════════════════════════\n";
        echo "📊 التقرير النهائي\n";
        echo "═══════════════════════════════════════════════════════════\n\n";
        
        $totalErrors = count($this->errors);
        $totalWarnings = count($this->warnings);
        
        if ($totalErrors == 0 && $totalWarnings == 0) {
            echo "🎉 ✅ جميع الفحوصات ناجحة!\n";
            echo "النظام جاهز للعمل بشكل كامل.\n\n";
        } else {
            if ($totalErrors > 0) {
                echo "❌ الأخطاء ($totalErrors):\n";
                foreach ($this->errors as $i => $error) {
                    echo "  " . ($i + 1) . ". $error\n";
                }
                echo "\n";
            }
            
            if ($totalWarnings > 0) {
                echo "⚠️ التحذيرات ($totalWarnings):\n";
                foreach ($this->warnings as $i => $warning) {
                    echo "  " . ($i + 1) . ". $warning\n";
                }
                echo "\n";
            }
        }
        
        echo "═══════════════════════════════════════════════════════════\n";
    }
}

// التشغيل
$verifier = new MigrationVerifier(CONFIG_FILE);
$verifier->verify();
